package com.qxt;

import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.NameValuePair;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.message.BasicNameValuePair;
import org.apache.hc.core5.http.io.entity.UrlEncodedFormEntity;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

/**
 * 企信通短信客户端（完整版）
 * 支持：发送短信、余额查询、状态报告、上行回复
 * 编码支持：UTF-8 / GBK
 */
public class SmsClient {

    // ========== 接口地址 ==========
    private final String SEND_URL_UTF8 = "http://cloud.thinksms.com/sms.aspx";
    private final String SEND_URL_GBK = "http://cloud.thinksms.com/smsGBK.aspx";
    private final String STATUS_URL = "http://cloud.thinksms.com/statusApi.aspx";
    private final String UPLINK_URL = "http://cloud.thinksms.com/callApi.aspx";
    private final String BALANCE_URL = "http://cloud.thinksms.com/sms.aspx"; // 共用发送接口

    // ========== 账号配置 ==========
    private final String userId;
    private final String account;
    private final String password;

    // ========== HTTP 客户端（连接池）==========
    private final CloseableHttpClient httpClient;

    public SmsClient(String userId, String account, String password) {
        this.userId = userId;
        this.account = account;
        this.password = password;
        this.httpClient = HttpClients.createDefault();
    }

    // ===================================================================================
    // 1. 发送短信
    // ===================================================================================

    public String sendSmsUtf8(String mobile, String content) throws IOException {
        return sendRequest(SEND_URL_UTF8, "send", mobile, content, StandardCharsets.UTF_8.name());
    }

    public String sendSmsGbk(String mobile, String content) throws IOException {
        return sendRequest(SEND_URL_GBK, "send", mobile, content, "GBK");
    }

    // ===================================================================================
    // 2. 余额及已发送量查询
    // ===================================================================================

    /**
     * 查询账户余额和发送总量
     * @return 原始 JSON 字符串
     */
    public String queryBalance() throws IOException {
        List<NameValuePair> params = new ArrayList<>();
        params.add(new BasicNameValuePair("action", "overage"));
        params.add(new BasicNameValuePair("rt", "json"));
        params.add(new BasicNameValuePair("userid", userId));
        params.add(new BasicNameValuePair("account", account));
        params.add(new BasicNameValuePair("password", password));

        return executePost(BALANCE_URL, params, StandardCharsets.UTF_8.name());
    }

    // ===================================================================================
    // 3. 状态报告获取（短信是否送达）
    // ===================================================================================

    /**
     * 获取状态报告
     * @param statusNum 每次获取的数量，默认4000
     * @return 原始 JSON 字符串
     */
    public String fetchStatusReport(int statusNum) throws IOException {
        List<NameValuePair> params = new ArrayList<>();
        params.add(new BasicNameValuePair("action", "query"));
        params.add(new BasicNameValuePair("rt", "json"));
        params.add(new BasicNameValuePair("userid", userId));
        params.add(new BasicNameValuePair("account", account));
        params.add(new BasicNameValuePair("password", password));
        params.add(new BasicNameValuePair("statusNum", String.valueOf(statusNum)));

        return executePost(STATUS_URL, params, StandardCharsets.UTF_8.name());
    }

    // 无参版本，默认取4000条
    public String fetchStatusReport() throws IOException {
        return fetchStatusReport(4000);
    }

    // ===================================================================================
    // 4. 上行短信获取（用户回复）
    // ===================================================================================

    /**
     * 获取用户上行回复（短信）
     * @param ownExt 是否使用自有扩展号：0或空=否，1=是
     * @return 原始 JSON 字符串
     */
    public String fetchUplinkMessages(String ownExt) throws IOException {
        List<NameValuePair> params = new ArrayList<>();
        params.add(new BasicNameValuePair("action", "query"));
        params.add(new BasicNameValuePair("rt", "json"));
        params.add(new BasicNameValuePair("userid", userId));
        params.add(new BasicNameValuePair("account", account));
        params.add(new BasicNameValuePair("password", password));
        if (ownExt != null && !ownExt.isEmpty()) {
            params.add(new BasicNameValuePair("ownExt", ownExt));
        }

        return executePost(UPLINK_URL, params, StandardCharsets.UTF_8.name());
    }

    // 无参版本
    public String fetchUplinkMessages() throws IOException {
        return fetchUplinkMessages("");
    }

    // ===================================================================================
    // 私有核心方法
    // ===================================================================================

    private String sendRequest(String url, String action, String mobile, String content, String charset) throws IOException {
        List<NameValuePair> params = new ArrayList<>();
        params.add(new BasicNameValuePair("action", action));
        params.add(new BasicNameValuePair("rt", "json"));
        params.add(new BasicNameValuePair("userid", userId));
        params.add(new BasicNameValuePair("account", account));
        params.add(new BasicNameValuePair("password", password));
        params.add(new BasicNameValuePair("mobile", mobile));
        params.add(new BasicNameValuePair("content", content));

        return executePost(url, params, charset);
    }

    private String executePost(String url, List<NameValuePair> params, String charset) throws IOException {
        try (CloseableHttpResponse response = executeHttpPost(url, params, charset)) {
            return EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
        }
    }

    private CloseableHttpResponse executeHttpPost(String url, List<NameValuePair> params, String charset) throws IOException {
        HttpPost post = new HttpPost(url);
        post.setEntity(new UrlEncodedFormEntity(params, ContentType.APPLICATION_FORM_URLENCODED.withCharset(charset)));
        return httpClient.execute(post);
    }

    // ===================================================================================
    // 资源释放
    // ===================================================================================

    public void close() throws IOException {
        if (httpClient != null) {
            httpClient.close();
        }
    }
}